package nl.oostnl.ventureplan.jobs.financial;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.log4j.Logger;
import org.apache.poi.hssf.usermodel.HSSFDateUtil;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.DataFormatter;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.eclipse.persistence.tools.file.FileUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import nl.buildersenperformers.xam.engine.Dataset;
import nl.buildersenperformers.xam.engine.DatasetException;
import nl.buildersenperformers.xam.engine.Operation;
import nl.buildersenperformers.xam.engine.OperationException;
import nl.buildersenperformers.xam.engine.Operator;
import nl.buildersenperformers.xam.engine.XamEngine;
import nl.buildersenperformers.xam.engine.dataset.jdbc.ListGeneric;
import nl.knowledgeplaza.util.ConfigurationProperties;
import nl.knowledgeplaza.util.Properties;
import nl.oostnl.ventureplan.jobs.model.VpDocument;

public class ImportFinancial implements Operation {
	/** Standard variable for determining version of a class file. */
	public static final String SOURCECODE_VERSION = "$Revision: 1.2 $";

	/** set up a Log4J category to log in */
	private static Logger log4j = nl.knowledgeplaza.util.Log4jUtil.createLogger();
	
	private Properties iProperties;
	private Map<String,Object> iParams;
	
	private XamEngine iXE=null;
	private Dataset iDataset;
	private Dataset iDocDataset;
	private String iCachePath;
	private Integer iDocId=null;
	private String iSessionId=null;
	
	/**
	 * 
	 */
	public ImportFinancial() {
		super();
		iXE=new XamEngine();
		try {
			iDocDataset=iXE.getDataset("Documenten");
		} catch (DatasetException e) {
			throw new RuntimeException(e);
		}
		
		String lCachePath=ConfigurationProperties.get().get("FINANCIAL.CachePath");
		if(lCachePath==null){
			log4j.error("FINANCIAL.CachePath not set in configuration");
		}
		iCachePath=lCachePath;
		if(log4j.isDebugEnabled()){
			log4j.debug("Financial import cache directory: " + iCachePath);
		}
	}

	@Override
	public Operator getOperator(String pName) {
		/* nothin for now*/
		return null;
	}

	@Override
	public void setParameter(String pName, Object pValue) throws OperationException {
		if(pName.equals("doc_id")){
			iDocId=(Integer)pValue;
		} else if(pName.equals("session_id")){
			iSessionId=(String)pValue;
		} else {
			iParams.put(pName, pValue);
		}

	}

	@Override
	public void setParameter(String pName, List<Object> pValue) {
		/* nothin for now*/

	}

	@Override
	public boolean canExecute() {
		return true;
	}

	@Override
	public boolean supportsResultset() {
		return false;
	}

	@Override
	public ResultSet executeAsResultset() throws OperationException {
		/* no support */
		return null;
	}

	@Override
	public Map<String, List<Object>> executeAsValueMap() throws OperationException {
		Map<String, List<Object>> lResult = new HashMap<>();
		List<Object> lCacheFileResult = new ArrayList<>();
		lResult.put("FinancialCacheFile", lCacheFileResult);
		
		//Check if cached file exists. if so return it.
		VpDocument doc=getDocument(iDocId);
		doc.setSessionId(iSessionId);
		if(log4j.isDebugEnabled()){
			log4j.debug("VpDocument: " + doc);
		}
			
		String cacheFileName=String.format("%s_%s.xml", doc.getDocId(),doc.getCurrentVersionId());
		File lCacheDir=new File(iCachePath);
		File lCacheFile=new File(lCacheDir,cacheFileName);
		if(log4j.isDebugEnabled()){
			log4j.debug("CacheDir: " + lCacheDir);
			log4j.debug("CacheFile: " + lCacheFile);
		}
		
		if(lCacheFile.exists()){
			//cache file exists return the file
			lCacheFileResult.add(lCacheFile.toURI().toString());
			return lResult;
		} 
		
		//check filename for excel
		if(!doc.getDocName().endsWith("xlsx") && !doc.getDocName().endsWith(".XLSX")){
			log4j.warn("Document " + doc.getDocName() + " is not excel!");
			lCacheFileResult.add("NO_EXCEL");
			return lResult;
		}
		
		//No cached file exists, import file, create cached file and return the path.
		File tmpDoc=null;
		try {
			tmpDoc=File.createTempFile(doc.getDocName(),".xlsxtmp",null);
			if(log4j.isDebugEnabled()){
				log4j.debug("TempFile: " + tmpDoc);
			}
		} catch (IOException e) {
			throw new OperationException(e);
		} 
		try(
				OutputStream tmpDocStream=new FileOutputStream(tmpDoc);
				InputStream orgFileStream=doc.getFileStream()
				){
			//get org file
			FileUtil.copy(orgFileStream, tmpDocStream);
			
			//import file
			importFile(tmpDoc,lCacheFile);
			
			lCacheFileResult.add(lCacheFile.toURI().toString());
			return lResult;
		} catch (IOException e) {
			throw new OperationException(e);
		} 
		
	}

	@Override
	public List<Map<String, Object>> executeAsValueList() throws OperationException {
		/* no support */
		return null;
	}

	@Override
	public void close() {
		/* nothin for now*/

	}

	@Override
	public void setProperties(Properties pProperties) {
		iProperties=pProperties;

	}

	@Override
	public void setDataset(Dataset pDataset) {
		iDataset = pDataset;

	}

	@Override
	public String getDescription() {
		return "Imports a financial excel sheet and returns the cahced result xml file";
	}
	
	public void importFile(File file, File outFile) throws OperationException{
		try(FileOutputStream out=new FileOutputStream(outFile); FileInputStream fileStream=new FileInputStream(file)) {
			
			
			//read file and find the Dashboard sheet
			XSSFWorkbook workbook = new XSSFWorkbook(fileStream);
			FormulaEvaluator evaluator = workbook.getCreationHelper().createFormulaEvaluator();
			Locale dutch=Locale.forLanguageTag("nl-NL");
			DataFormatter formatter = new DataFormatter(dutch);
			
			int dashboardPos = workbook.getSheetIndex("Dashboard");
			if (dashboardPos==-1){
				//Dashboard sheet not found
				throw new OperationException("Dashboard sheet not found in file " + file.getAbsolutePath());
			}
			XSSFSheet sheet = workbook.getSheetAt(dashboardPos);
			//Iterate through each rows one by one
			
			DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
	        DocumentBuilder docBuilder = docFactory.newDocumentBuilder();
	        
	        Document doc = docBuilder.newDocument();
	        Element rootElement = doc.createElement("table");
	        
	        doc.appendChild(rootElement);
			
			for(int i=9;i<=25;i++){
				//read rows 10 to 16
				Row row = sheet.getRow(i);
				
				Element rowElem=doc.createElement("row");
				rootElement.appendChild(rowElem);
				
				for(int j=0;j<=10;j++){
					if(j==1)continue;
					Cell cell=row.getCell(j,Row.MissingCellPolicy.CREATE_NULL_AS_BLANK);
					
					
					if(cell==null) continue;
					int colspan=getCollspan(cell, sheet);
					
					if(cell.getCellType()==CellType.BLANK && colspan> 1) continue;
					
					Element cellElem=doc.createElement("cell");
					rowElem.appendChild(cellElem);
					
					
					cellElem.setAttribute("colspan", String.valueOf(colspan));
					
					StringBuilder classStr=new StringBuilder();
					int fontIndex=cell.getCellStyle().getFontIndexAsInt();
					XSSFFont font=workbook.getFontAt(fontIndex);
					
					if(font.getBold()){
						classStr.append("bold ");
					}
					if(font.getItalic()){
						classStr.append("italic");
					}
					
					switch(cell.getCellStyle().getAlignment()){
					case CENTER: classStr.append("align-center "); break;
					case RIGHT:  classStr.append("align-right "); break;
					case CENTER_SELECTION: classStr.append("align-center "); break;
					default:
						if(cell.getCellType()==CellType.NUMERIC || (cell.getCellType()==CellType.FORMULA && cell.getCachedFormulaResultType()==CellType.NUMERIC)){
							classStr.append("align-right ");
						}
						break;
					}
					
					Element value=doc.createElement("value");
					cellElem.appendChild(value);
					
					if(cell.getCellType()==CellType.FORMULA){
						value.setTextContent(formatter.formatCellValue(cell, evaluator));
						
					} else {
						value.setTextContent(formatter.formatCellValue(cell));
					}
					/*switch (cell.getCellType()) 
                    {
                        case NUMERIC:
                            System.out.println(" N:" + cell.getAddress().formatAsString() + "-" +  cell.getNumericCellValue());
                            value.setTextContent(textContent);
                            break;
                        case STRING:
                            System.out.println(" S:" + cell.getAddress().formatAsString() + "-" +  cell.getStringCellValue());
                            break;
                        case FORMULA:
                        	switch (cell.getCachedFormulaResultType()) {
                            case BOOLEAN:
                                System.out.println("FS:" + cell.getAddress().formatAsString() + "-" +  cell.getBooleanCellValue());
                                break;
                            case NUMERIC:
                            	if(HSSFDateUtil.isCellDateFormatted(cell)){
                            		Date date = HSSFDateUtil.getJavaDate(cell.getNumericCellValue());
                            		System.out.println("FD:" + cell.getAddress().formatAsString() + "-" +  date);
                            	} else {
                            		System.out.println("FN:" + cell.getAddress().formatAsString() + "-" +  cell.getNumericCellValue());
                            	}
                                break;
                            case STRING:
                                System.out.println("FS:" + cell.getAddress().formatAsString() + "-" +  cell.getRichStringCellValue());
                                break;
							default:
								break;
                        	}
                            break;
                        default:
                        	System.out.println(cell.getAddress().formatAsString() + "-TYPE:" + cell.getCellType().toString() );
                    }*/
					
					cellElem.setAttribute("class", classStr.toString());
                }
				
				
			}
			
			
			//write xml
			
			TransformerFactory transformerFactory = TransformerFactory.newInstance();
			
	        Transformer transformer = transformerFactory.newTransformer();
	        
	        if(log4j.isDebugEnabled()){
		        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
		        transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", String.valueOf(4));
	        }
	        DOMSource source = new DOMSource(doc);
	        StreamResult result = new StreamResult(out);

	        transformer.transform(source, result);
		} catch (IOException e) {
			log4j.error("Could not open file: "+ file.getAbsoluteFile(),e);
			throw new OperationException("Could not open file: "+ file.getAbsoluteFile(),e);
		} catch (ParserConfigurationException | TransformerException e) {
			log4j.error("Could not write to file: " + outFile.getAbsoluteFile(),e);
			throw new OperationException("Could not write to file: " + outFile.getAbsoluteFile(),e);
		} 
		
	}
	
	private int getCollspan(Cell cell, XSSFSheet sheet ){
		
		for (CellRangeAddress region : sheet.getMergedRegions()) {
			if(region.isInRange(cell)){
				return 1 + region.getLastColumn() - region.getFirstColumn();
			}
		}
		
		return 1;
	}
	
	public VpDocument getDocument(Integer pDocId) throws OperationException  {
		XamEngine lXE=new XamEngine();
		try {
			Dataset lDS= lXE.getDataset("Documenten");
			Operation lOper=lDS.getOperation("GetDocument");
			lOper.setParameter("doc_id", pDocId);
			ListGeneric<VpDocument> lDocOper=new ListGeneric<VpDocument>(lOper){};
			List<VpDocument> docs=lDocOper.execute();
			
			if(docs==null || docs.isEmpty()) {
				throw new OperationException("Doc not found");
			}
			
			return docs.get(0);
		} catch (DatasetException e) {
			throw new OperationException(e);
		}
		
	}

}
