package nl.oostnl.ventureplan.jobs.onboarding.adapter;

import java.sql.Timestamp;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import nl.knowledgeplaza.util.Properties;
import nl.buildersenperformers.xam.engine.Dataset;
import nl.buildersenperformers.xam.engine.DatasetException;
import nl.buildersenperformers.xam.engine.Operation;
import nl.buildersenperformers.xam.engine.OperationException;
import nl.buildersenperformers.xam.engine.XamEngine;
import nl.oostnl.ventureplan.jobs.onboarding.domain.AanvraagQueueItem;

public class QueueDao {
    private final Properties properties;

    public QueueDao(Properties properties) {
        this.properties = properties;
    }

    public List<AanvraagQueueItem> fetchNextBatch(int limit) {
        List<AanvraagQueueItem> items = new ArrayList<AanvraagQueueItem>();
        try {
            XamEngine xe = new XamEngine();
            Dataset ds = xe.getDataset("Aanvraag");
            Operation op = ds.getOperation("QueueList");
            op.setProperties(properties);
            op.setParameter("limit", limit);
            Map<String, List<Object>> vm = op.executeAsValueMap();
            if (vm == null || vm.isEmpty()) {
                return items;
            }
            int rowCount = 0;
            // bepaal aantal rijen via een kolom (queue_id altijd aanwezig)
            List<Object> qids = vm.get("queue_id");
            if (qids != null) {
                rowCount = qids.size();
            }
            for (int i = 0; i < rowCount; i++) {
                long queueId = toLong(vm.get("queue_id").get(i));
                long aanvraagTypeId = toLong(vm.get("aanvraag_type_id").get(i));
                String requesterEmail = toString(vm.get("requester_email").get(i));
                String requesterKvk = vm.containsKey("requester_kvk") && vm.get("requester_kvk").size() > i ? toString(vm.get("requester_kvk").get(i)) : null;
                String payloadJson = toString(vm.get("payload_json").get(i));
                OffsetDateTime createdAt = toOffsetDateTime(vm.get("created_at").get(i));
                items.add(new AanvraagQueueItem(queueId, aanvraagTypeId, requesterEmail, requesterKvk, payloadJson, createdAt));
            }
            return items;
        } catch (DatasetException e) {
            throw new RuntimeException(e);
        } catch (OperationException e) {
            throw new RuntimeException(e);
        }
    }

    public void markCompleted(long queueId) {
        updateStatus(queueId, "completed");
    }

    public void markRejected(long queueId, String errorCode, String errorDetails) {
        updateStatus(queueId, "rejected", errorCode, errorDetails);
    }

    private void updateStatus(long queueId, String status) {
        try {
            XamEngine xe = new XamEngine();
            Dataset ds = xe.getDataset("Aanvraag");
            Operation op = ds.getOperation("QueueStatusUpdate");
            op.setProperties(properties);
            op.setParameter("queue_id", queueId);
            op.setParameter("status", status);
            op.executeAsValueMap();
        } catch (DatasetException e) {
            throw new RuntimeException(e);
        } catch (OperationException e) {
            throw new RuntimeException(e);
        }
    }

    private void updateStatus(long queueId, String status, String errorCode, String errorDetails) {
        try {
            XamEngine xe = new XamEngine();
            Dataset ds = xe.getDataset("Aanvraag");
            Operation op = ds.getOperation("QueueStatusUpdate");
            op.setProperties(properties);
            op.setParameter("queue_id", queueId);
            op.setParameter("status", status);
            if (errorCode != null) op.setParameter("error_code", errorCode);
            if (errorDetails != null) op.setParameter("error_details", errorDetails);
            op.executeAsValueMap();
        } catch (DatasetException e) {
            throw new RuntimeException(e);
        } catch (OperationException e) {
            throw new RuntimeException(e);
        }
    }

    private long toLong(Object o) {
        if (o == null) return 0L;
        if (o instanceof Number) return ((Number) o).longValue();
        try { return Long.parseLong(String.valueOf(o)); } catch (Exception e) { return 0L; }
    }

    private String toString(Object o) {
        return o == null ? null : String.valueOf(o);
    }

    private OffsetDateTime toOffsetDateTime(Object o) {
        if (o == null) return null;
        if (o instanceof Timestamp) {
            Timestamp ts = (Timestamp) o;
            return OffsetDateTime.ofInstant(ts.toInstant(), ZoneOffset.UTC);
        }
        // fallback: try parse string
        try {
            return OffsetDateTime.parse(String.valueOf(o));
        } catch (Exception e) {
            return null;
        }
    }
}
