package nl.oostnl.ventureplan.jobs.onboarding.service;

import nl.oostnl.ventureplan.jobs.onboarding.domain.AanvraagQueueItem;
import nl.oostnl.ventureplan.jobs.onboarding.domain.ProcessingResult;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.JsonNode;
import nl.oostnl.ventureplan.jobs.onboarding.domain.IntakeData;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Optional;
import java.util.List;
import java.util.regex.Pattern;

import nl.knowledgeplaza.util.Properties;
import nl.buildersenperformers.xam.engine.XamEngine;
import nl.buildersenperformers.xam.engine.Dataset;
import nl.buildersenperformers.xam.engine.Operation;
import nl.buildersenperformers.xam.engine.OperationException;
import nl.buildersenperformers.xam.engine.DatasetException;

import org.apache.log4j.Logger;

/**
 * Verwerkt één onboarding queue-item.
 *
 * Stappen (high-level):
 * 1) Payload parsen
 * 2) Validaties (e-mail syntaxis, uniciteit in gen_ug, KvK-formaat)
 * 3) (Volgende stap) Aanvraagverwerking via DB-functie (aanvraag + contact + workflow)
 */
public class ItemProcessor {
    /** set up a Log4J category to log in */
    private static Logger log4j = nl.knowledgeplaza.util.Log4jUtil.createLogger();

    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final Pattern EMAIL_REGEX = Pattern.compile("^[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\\.[A-Za-z]{2,}$");

    private final Properties properties;

    public ItemProcessor(Properties properties) {
        this.properties = properties;
    }

    /** Valideer e-mailadres (required, regex) en check op bestaan in gen_ug. Retourneer REJECTED of null bij succes. */
    private ProcessingResult validateEmailOrReject(AanvraagQueueItem item, IntakeData intake) {
        String email = intake == null ? null : intake.getEmail();
        if (email == null || email.trim().isEmpty()) {
            log4j.warn("Validation failed: EMAIL_REQUIRED for queue_id=" + (item == null ? null : item.getQueueId()));
            return new ProcessingResult(ProcessingResult.Status.REJECTED, "EMAIL_REQUIRED", "E-mail is verplicht", 0L);
        }
        if (!EMAIL_REGEX.matcher(email).matches()) {
            log4j.warn("Validation failed: EMAIL_INVALID for queue_id=" + (item == null ? null : item.getQueueId()) + ", email=" + email);
            return new ProcessingResult(ProcessingResult.Status.REJECTED, "EMAIL_INVALID", "E-mail formaat ongeldig", 0L);
        }
        if (userExistsByEmail(email)) {
            log4j.info("Validation failed: EMAIL_EXISTS for queue_id=" + (item == null ? null : item.getQueueId()) + ", email=" + email);
            return new ProcessingResult(ProcessingResult.Status.REJECTED, "EMAIL_EXISTS", "E-mailadres is al in gebruik als loginnaam", 0L);
        }
        return null;
    }

    public ProcessingResult process(AanvraagQueueItem item) {
        // Start logging met kerncontext
        if (log4j.isInfoEnabled() && item != null) {
            log4j.info("Start processing queue_id=" + item.getQueueId() + 
                       ", aanvraag_type_id=" + item.getAanvraagTypeId() +
                       ", email=" + item.getRequesterEmail());
        }
        // 1) Parse payloadJson uit item naar een IntakeData POJO (JSON -> object)
        JsonNode payload = parseJsonNode(item == null ? null : item.getPayloadJson());
        if (item != null && item.getPayloadJson() != null && payload == null) {
            log4j.warn("Invalid JSON payload for queue_id=" + (item == null ? null : item.getQueueId()));
            try { updateQueueStatus(item, "rejected", "INVALID_JSON", "Payload kon niet worden geparsed"); } catch (Exception ex) { log4j.error("Status update failed", ex); }
            return new ProcessingResult(ProcessingResult.Status.REJECTED, "INVALID_JSON", "Payload kon niet worden geparsed", 0L);
        }
        IntakeData intake = buildIntakeData(item, payload);
        if (log4j.isDebugEnabled()) {
            log4j.debug("IntakeData: " + intake);
        }
        // 2) Valideer verplichte velden (bijv. e‑mail, type-id, consent). Bij fout: return REJECTED met errorMessage
        ProcessingResult emailValidation = validateEmailOrReject(item, intake);
        if (emailValidation != null) {
            try { updateQueueStatus(item, "rejected", emailValidation.getErrorCode(), emailValidation.getErrorMessage()); } catch (Exception ex) { log4j.error("Status update failed", ex); }
            return emailValidation;
        }

        String kvk = intake.getKvk().orElse(null);
        if (kvk != null && !kvk.isEmpty() && !kvk.matches("^[0-9]{8}$")) {
            log4j.warn("Validation failed: INVALID_KVK for queue_id=" + (item == null ? null : item.getQueueId()) + ", kvk=" + kvk);
            try { updateQueueStatus(item, "rejected", "INVALID_KVK", "KvK moet 8 cijfers bevatten"); } catch (Exception ex) { log4j.error("Status update failed", ex); }
            return new ProcessingResult(ProcessingResult.Status.REJECTED, "INVALID_KVK", "KvK moet 8 cijfers bevatten", 0L);
        }
        // 3) Bepaal type-specifieke logica (op basis van aanvraagTypeId). Geen DB‑calls hier; puur businessregels
        // 4) Voer de verwerking uit (bijv. mappen/normaliseren), en bouw optionele metadata (bv. counts, ids)
        // 5) Bij succes: return SUCCESS met evt. metadata; Bij exception: return REJECTED met errorMessage
        if (log4j.isInfoEnabled() && item != null) {
            log4j.info("Validations passed for queue_id=" + item.getQueueId());
        }
        try {
            XamEngine xe = new XamEngine();
            Dataset ds = xe.getDataset("Klantportaal");
            Operation op = ds.getOperation("ProcessQueueItem");
            op.setProperties(properties);
            op.setParameter("queue_id", item.getQueueId());
            Map<String, List<Object>> vm = op.executeAsValueMap();
            if (log4j.isInfoEnabled()) {
                log4j.info("Aanvraag verwerkt voor queue_id=" + item.getQueueId() + ", resultKeys=" + (vm == null ? null : vm.keySet()));
            }
            try { updateQueueStatus(item, "completed", null, null); } catch (Exception ex) { log4j.error("Status update failed", ex); }
            return new ProcessingResult(ProcessingResult.Status.SUCCESS, null, null, 0L);
        } catch (OperationException e) {
            String code = extractErrorCode(e == null ? null : e.getMessage());
            log4j.error("Aanvraagverwerking fout (operation)", e);
            try { updateQueueStatus(item, "error", code == null ? "PROCESS_FAILED" : code, e.getMessage()); } catch (Exception ex) { log4j.error("Status update failed", ex); }
            return new ProcessingResult(ProcessingResult.Status.ERROR, code == null ? "PROCESS_FAILED" : code, e.getMessage(), 0L);
        } catch (DatasetException e) {
            String code = extractErrorCode(e == null ? null : e.getMessage());
            log4j.error("Aanvraagverwerking fout (dataset)", e);
            try { updateQueueStatus(item, "error", code == null ? "PROCESS_FAILED" : code, e.getMessage()); } catch (Exception ex) { log4j.error("Status update failed", ex); }
            return new ProcessingResult(ProcessingResult.Status.ERROR, code == null ? "PROCESS_FAILED" : code, e.getMessage(), 0L);
        }
    }

    /** Parse JSON payload naar JsonNode. */
    private JsonNode parseJsonNode(String json) {
        if (json == null) return null;
        try {
            return MAPPER.readTree(json);
        } catch (Exception e) {
            log4j.error("Payload kon niet worden geparsed", e);
            return null;
        }
    }

    /** Bouw IntakeData uit queue-kolommen en volledige payload (als answers-map). */
    private IntakeData buildIntakeData(AanvraagQueueItem item, JsonNode payload) {
        String email = item == null ? null : item.getRequesterEmail();
        String kvk = item == null ? null : item.getRequesterKvk();
        Map<String, String> answers = payloadToMap(payload);
        return new IntakeData(email, Optional.ofNullable(kvk), answers);
    }

    /** Converteer vlakke JSON-object payload naar Map<String,String>. */
    private Map<String, String> payloadToMap(JsonNode node) {
        Map<String, String> out = new HashMap<String, String>();
        if (node == null || !node.isObject()) return out;
        Iterator<Map.Entry<String, JsonNode>> it = node.fields();
        while (it.hasNext()) {
            Map.Entry<String, JsonNode> e = it.next();
            JsonNode v = e.getValue();
            String val = v == null || v.isNull() ? null : v.asText();
            out.put(e.getKey(), val);
        }
        return out;
    }

    /** Check of loginnaam (ug_name) al bestaat in gen_ug via Admin/UserExistsByName. */
    private boolean userExistsByEmail(String email) {
        try {
            XamEngine xe = new XamEngine();
            Dataset ds = xe.getDataset("Admin");
            Operation op = ds.getOperation("UserExistsByName");
            op.setProperties(properties);
            op.setParameter("ug_name", email);
            Map<String, List<Object>> vm = op.executeAsValueMap();
            if (vm == null || !vm.containsKey("exists") || vm.get("exists").isEmpty()) return false;
            Object val = vm.get("exists").get(0);
            String s = val == null ? null : String.valueOf(val);
            return "true".equalsIgnoreCase(s);
        } catch (OperationException | DatasetException e) {
            log4j.error("Kon user-exists check niet uitvoeren", e);
            return false;
        }
    }

    private void updateQueueStatus(AanvraagQueueItem item, String status, String errorCode, String errorDetails) throws OperationException, DatasetException {
        if (item == null) return;
        XamEngine xe = new XamEngine();
        Dataset ds = xe.getDataset("Aanvraag");
        Operation op = ds.getOperation("QueueStatusUpdate");
        op.setProperties(properties);
        op.setParameter("queue_id", item.getQueueId());
        op.setParameter("status", status);
        if (errorCode != null) op.setParameter("error_code", errorCode);
        if (errorDetails != null) op.setParameter("error_details", errorDetails);
        op.executeAsValueMap();
    }

    private String extractErrorCode(String message) {
        if (message == null) return null;
        String m = message.trim();
        int sp = m.indexOf(' ');
        String head = sp > 0 ? m.substring(0, sp) : m;
        if (head.matches("[A-Z0-9_]+")) return head;
        return null;
    }

}
